<?php declare(strict_types=1);

namespace FacturaScripts\Plugins\PartesSemanales\Controller;

use Closure;
use FacturaScripts\Core\Base\Controller;
use FacturaScripts\Core\Base\DataBase\DataBaseWhere;
use FacturaScripts\Dinamic\Model\CodeModel;
use FacturaScripts\Plugins\PartesSemanales\Model\Aviso;
use FacturaScripts\Plugins\PartesSemanales\Model\AvisoParte;
use FacturaScripts\Plugins\PartesSemanales\Model\AvisoParteChecklist;
use FacturaScripts\Plugins\PartesSemanales\Model\AvisoParteMaterial;
use FacturaScripts\Plugins\PartesSemanales\Model\AvisoParteKilometraje;
use FacturaScripts\Plugins\PartesSemanales\Model\AvisoParteIncidencia;
use FacturaScripts\Core\Session;
use FacturaScripts\Dinamic\Model\AttachedFile;

class EditAviso extends Controller
{
    /** @var Aviso */
    private $aviso;

    /** @var AvisoParte[] */
    private $partes = [];

    /** @var AvisoParteMaterial[] */
    private $materiales = [];

    /** @var AvisoParteKilometraje[] */
    private $kilometrajes = [];

    /** @var AvisoParteIncidencia[] */
    private $incidencias = [];

    /** @var AvisoParteChecklist[] */
    private $checklists = [];

    /** @var CodeModel */
    public $codeModel;

    public function getModelClassName(): string
    {
        return 'Aviso';
    }

    public function getModel(): Aviso
    {
        return $this->aviso;
    }

    public function getPartes(): array
    {
        return $this->partes;
    }

    public function getMateriales(): array
    {
        return $this->materiales;
    }

    public function getKilometrajes(): array
    {
        return $this->kilometrajes;
    }

    public function getIncidencias(): array
    {
        return $this->incidencias;
    }

    public function getChecklists(): array
    {
        return $this->checklists;
    }

    public function getPageData(): array
    {
        $pageData = parent::getPageData();
        $pageData['title'] = 'Aviso';
        $pageData['icon'] = 'fas fa-bullhorn';
	    $pageData['menu'] = "";
        return $pageData;
    }

    public function createViews(): Closure
    {
        return function () {
            $viewName = 'EditAviso';
            $pageData = $this->getPageData();
            $this->addHtmlView($viewName, $viewName, 'Aviso', $pageData['title'], $pageData['icon']);
            $this->setSettings($viewName, 'btnPrint', true);
        };
    }

    public function privateCore(&$response, $user, $permissions): void
    {
        parent::privateCore($response, $user, $permissions);

        $this->codeModel = new CodeModel();

        $this->aviso = new Aviso();
        $this->aviso->loadFromCode($this->request->get('code'));

        $action = $this->request->request->get('action');
        if (in_array($action, ['insert', 'edit'], true)) {
            $this->aviso->nombre = $this->request->request->get('nombre');
            $this->aviso->direccion = $this->request->request->get('direccion');
            $this->aviso->save();

            if ($action === 'insert') {
                $this->redirect($this->aviso->url());
            }
        } elseif ($action === 'saveParte' && $this->aviso->id) {
            $parte = new AvisoParte();
            $idparte = (int) $this->request->request->get('idparte');
            if ($idparte) {
                $parte->loadFromCode($idparte);
                $parte->idaviso = $this->aviso->id;
            } else {
                $parte->idaviso = $this->aviso->id;
            }
            if ($this->request->request->has('horas')) {
                $parte->horas = (float) $this->request->request->get('horas');
            }
            if ($this->request->request->has('horas_extra')) {
                $parte->horas_extra = (float) $this->request->request->get('horas_extra');
            }
            if ($this->request->request->has('comidas')) {
                $parte->comidas = (int) $this->request->request->get('comidas');
            }
            if ($this->request->request->has('dietas')) {
                $parte->dietas = (float) $this->request->request->get('dietas');
            }
            if ($this->request->request->has('checklist')) {
                $parte->checklist = $this->request->request->get('checklist');
            }
            $parte->save();

            if (is_array($this->request->request->get('checklist_nuevos'))) {
                foreach ($this->request->request->get('checklist_nuevos') as $desc) {
                    $desc = trim((string) $desc);
                    if ('' === $desc) {
                        continue;
                    }
                    $check = new AvisoParteChecklist();
                    $check->idavisoparte = $parte->id;
                    $check->descripcion = $desc;
                    $check->marcado = false;
                    $check->save();
                }
            }

            if (
                $this->request->request->has('material_descripcion')
                || $this->request->request->has('material_cantidad')
                || (isset($_FILES['material_files']) && is_array($_FILES['material_files']['tmp_name']))
            ) {
                $material = new AvisoParteMaterial();
                $material->idavisoparte = $parte->id;
                if ($this->request->request->has('material_cantidad')) {
                    $material->cantidad = (float) $this->request->request->get('material_cantidad');
                }
                if ($this->request->request->has('material_descripcion')) {
                    $material->descripcion = $this->request->request->get('material_descripcion');
                }
                $material->save();

                if (isset($_FILES['material_files']) && is_array($_FILES['material_files']['tmp_name'])) {
                    $destDir = FS_FOLDER . '/MyFiles/avisos/';
                    if (!is_dir($destDir)) {
                        mkdir($destDir, 0775, true);
                    }
                    foreach ($_FILES['material_files']['tmp_name'] as $key => $tmpName) {
                        if (!is_uploaded_file($tmpName)) {
                            continue;
                        }
                        $nombreArchivo = $_FILES['material_files']['name'][$key];
                        $safeName = preg_replace('/[^a-zA-Z0-9\._-]/', '_', $nombreArchivo);
                        $fileName = uniqid('aviso_') . '_' . $safeName;
                        $nombreLimpio = preg_replace('/[^a-zA-Z0-9]+/', '_', $nombreArchivo);
                        $nombreLimpio = trim($nombreLimpio, '_');
                        if (move_uploaded_file($tmpName, $destDir . $fileName)) {
                            $adj = new AttachedFile();
                            $adj->idmaterial = $material->id;
                            $adj->path = 'avisos/' . $fileName;
                            $adj->filename = $nombreLimpio;
                            $adj->save();
                        }
                    }
                }
            }

            if ($this->request->request->get('vehiculo')) {
                $kil = new AvisoParteKilometraje();
                $kil->idavisoparte = $parte->id;
                $kil->vehiculo = $this->request->request->get('vehiculo');
                $kil->kminicial = (int) $this->request->request->get('kminicial');
                $kil->kmfinal = (int) $this->request->request->get('kmfinal');
                $kil->kilometraje = max(0, $kil->kmfinal - $kil->kminicial);
                $kil->save();
            }

            if ($this->request->request->has('incidencia')) {
                $texto = $this->request->request->get('incidencia');
                if (!empty($texto)) {
                    $inc = new AvisoParteIncidencia();
                    $inc->idavisoparte = $parte->id;
                    $inc->descripcion = $texto;
                    $inc->save();
                }
            }

            $this->redirect($this->aviso->url());
        } elseif ($action === 'nuevoChecklistParte') {
            $this->nuevoChecklistParte();
        } elseif ($action === 'guardarChecklistParte') {
            $this->guardarChecklistParte();
        } elseif ($action === 'eliminarChecklistParte') {
            $this->eliminarChecklistParte();
        } elseif ($action === 'eliminarMaterial') {
            $this->eliminarMaterial();
        } elseif ($action === 'eliminarKilometraje') {
            $this->eliminarKilometraje();
        } elseif ($action === 'eliminarIncidencia') {
            $this->eliminarIncidencia();
        } elseif ($action === 'guardarIncidencia') {
            $this->guardarIncidencia();
        } elseif ($action === 'editarMaterial') {
            $this->editarMaterial();
        } elseif ($action === 'editarKilometraje') {
            $this->editarKilometraje();
        } elseif ($action === 'editarIncidencia') {
            $this->editarIncidencia();
        }

        $parteModel = new AvisoParte();
        if ($this->aviso->id) {
            $this->partes = $parteModel->all([new DataBaseWhere('idaviso', $this->aviso->id)]);

            $matModel = new AvisoParteMaterial();
            $kilModel = new AvisoParteKilometraje();
            $incModel = new AvisoParteIncidencia();
            $checkModel = new AvisoParteChecklist();
            $adjModel = new AttachedFile();

            foreach ($this->partes as $parte) {
                foreach ($matModel->all([new DataBaseWhere('idavisoparte', $parte->id)]) as $mat) {
                    $mat->parte = $parte;
                    $mat->files = $adjModel->all([new DataBaseWhere('idmaterial', $mat->id)]);
                    $this->materiales[] = $mat;
                }
                foreach ($kilModel->all([new DataBaseWhere('idavisoparte', $parte->id)]) as $kil) {
                    $kil->parte = $parte;
                    $this->kilometrajes[] = $kil;
                }
                foreach ($incModel->all([new DataBaseWhere('idavisoparte', $parte->id)]) as $inc) {
                    $inc->parte = $parte;
                    $inc->files = $adjModel->all([new DataBaseWhere('idincidencia', $inc->id)]);
                    $this->incidencias[] = $inc;
                }
                foreach ($checkModel->all([new DataBaseWhere('idavisoparte', $parte->id)]) as $check) {
                    $check->parte = $parte;
                    $this->checklists[] = $check;
                }
            }
        }
    }

    private function nuevoChecklistParte(): void
    {
        $idParte = (int) $this->request->request->get('idparte');
        if ($idParte <= 0) {
            return;
        }
        $check = new AvisoParteChecklist();
        $check->idavisoparte = $idParte;
        $check->descripcion = $this->request->request->get('descripcion');
        $check->marcado = false;
        $check->save();
    }

    private function guardarChecklistParte(): void
    {
        $check = new AvisoParteChecklist();
        $check->loadFromCode($this->request->request->get('id'));
        if ($check->nick !== (Session::get('user')->nick ?? null)) {
            return;
        }
        if ($this->request->request->has('descripcion')) {
            $check->descripcion = $this->request->request->get('descripcion');
        }
        $check->marcado = $this->request->request->get('marcado') ? true : false;
        $check->save();
    }

    private function eliminarChecklistParte(): void
    {
        $check = new AvisoParteChecklist();
        $check->loadFromCode($this->request->request->get('id'));
        if ($check->nick !== (Session::get('user')->nick ?? null)) {
            return;
        }
        $check->delete();
    }

    private function eliminarMaterial(): void
    {
        $material = new AvisoParteMaterial();
        $material->loadFromCode($this->request->request->get('id'));
        $material->delete();
    }

    private function eliminarKilometraje(): void
    {
        $kil = new AvisoParteKilometraje();
        $kil->loadFromCode($this->request->request->get('id'));
        $kil->delete();
    }

    private function eliminarIncidencia(): void
    {
        $inc = new AvisoParteIncidencia();
        $inc->loadFromCode($this->request->request->get('id'));
        if ($inc->nick !== (Session::get('user')->nick ?? null)) {
            return;
        }
        $inc->delete();
    }

    private function editarMaterial(): void
    {
        $material = new AvisoParteMaterial();
        $material->loadFromCode($this->request->request->get('id'));
        if (!$material->id) {
            return;
        }
        if ($this->request->request->has('cantidad')) {
            $material->cantidad = (float) $this->request->request->get('cantidad');
        }
        if ($this->request->request->has('descripcion')) {
            $material->descripcion = $this->request->request->get('descripcion');
        }
        $material->save();

        $deleteFiles = $this->request->request->get('delete_files', []);
        if (is_array($deleteFiles)) {
            foreach ($deleteFiles as $idFile) {
                $file = new AttachedFile();
                if ($file->loadFromCode($idFile)) {
                    $path = FS_FOLDER . '/MyFiles/' . $file->path;
                    if (file_exists($path)) {
                        unlink($path);
                    }
                    $file->delete();
                }
            }
        }

        if (isset($_FILES['files']) && is_array($_FILES['files']['tmp_name'])) {
            $destDir = FS_FOLDER . '/MyFiles/avisos/';
            if (!is_dir($destDir)) {
                mkdir($destDir, 0775, true);
            }
            foreach ($_FILES['files']['tmp_name'] as $key => $tmpName) {
                if (!is_uploaded_file($tmpName)) {
                    continue;
                }
                $nombreArchivo = $_FILES['files']['name'][$key];
                $safeName = preg_replace('/[^a-zA-Z0-9\._-]/', '_', $nombreArchivo);
                $fileName = uniqid('aviso_') . '_' . $safeName;
                $nombreLimpio = preg_replace('/[^a-zA-Z0-9]+/', '_', $nombreArchivo);
                $nombreLimpio = trim($nombreLimpio, '_');
                if (move_uploaded_file($tmpName, $destDir . $fileName)) {
                    $adj = new AttachedFile();
                    $adj->idmaterial = $material->id;
                    $adj->path = 'avisos/' . $fileName;
                    $adj->filename = $nombreLimpio;
                    $adj->save();
                }
            }
        }
    }

    private function editarKilometraje(): void
    {
        $kil = new AvisoParteKilometraje();
        $kil->loadFromCode($this->request->request->get('id'));
        if (!$kil->id) {
            return;
        }
        $kil->vehiculo = $this->request->request->get('vehiculo');
        $kil->kminicial = (int) $this->request->request->get('kminicial');
        $kil->kmfinal = (int) $this->request->request->get('kmfinal');
        $kil->kilometraje = max(0, $kil->kmfinal - $kil->kminicial);
        $kil->save();
    }

    private function editarIncidencia(): void
    {
        $inc = new AvisoParteIncidencia();
        $inc->loadFromCode($this->request->request->get('id'));
        if ($inc->nick !== (Session::get('user')->nick ?? null)) {
            return;
        }
        if ($this->request->request->has('titulo')) {
            $inc->titulo = $this->request->request->get('titulo');
        }
        if ($this->request->request->has('descripcion')) {
            $inc->descripcion = $this->request->request->get('descripcion');
        }
        $inc->save();

        $deleteFiles = $this->request->request->get('delete_files', []);
        if (is_array($deleteFiles)) {
            foreach ($deleteFiles as $idFile) {
                $file = new AttachedFile();
                $file->loadFromCode($idFile);
                if ($file->idincidencia == $inc->id) {
                    $file->delete();
                }
            }
        }

        if (!isset($_FILES['files']) || !is_array($_FILES['files']['tmp_name'])) {
            return;
        }

        $carpetaDestino = FS_FOLDER . '/MyFiles/';
        foreach ($_FILES['files']['tmp_name'] as $key => $tmp_name) {
            $nombreArchivo = $_FILES['files']['name'][$key];
            $rutaArchivo = $carpetaDestino . $nombreArchivo;

            $nombreLimpio = preg_replace('/[^a-zA-Z0-9]+/', '_', $nombreArchivo);
            $nombreLimpio = trim($nombreLimpio, '_');

            if (move_uploaded_file($tmp_name, $rutaArchivo)) {
                $adj = new AttachedFile();
                $adj->idincidencia = $inc->id;
                $adj->path = $nombreArchivo;
                $adj->filename = $nombreLimpio;
                $adj->save();
            }
        }
    }

    private function guardarIncidencia(): void
    {
        if (!isset($_REQUEST['action']) || 'guardarIncidencia' !== $_REQUEST['action']) {
            return;
        }

        $idParte = (int) $this->request->request->get('idparte');
        if ($idParte <= 0) {
            return;
        }

        $inc = new AvisoParteIncidencia();
        $inc->idavisoparte = $idParte;
        $inc->titulo = $this->request->request->get('titulo');
        $inc->descripcion = $this->request->request->get('descripcion');
        $inc->save();

        if (!isset($_FILES['files']) || !is_array($_FILES['files']['tmp_name'])) {
            return;
        }

        $carpetaDestino = FS_FOLDER . '/MyFiles/';
        foreach ($_FILES['files']['tmp_name'] as $key => $tmp_name) {
            $nombreArchivo = $_FILES['files']['name'][$key];
            $rutaArchivo = $carpetaDestino . $nombreArchivo;

            $nombreLimpio = preg_replace('/[^a-zA-Z0-9]+/', '_', $nombreArchivo);
            $nombreLimpio = trim($nombreLimpio, '_');

            if (move_uploaded_file($tmp_name, $rutaArchivo)) {
                $adj = new AttachedFile();
                $adj->idincidencia = $inc->id;
                $adj->path = $nombreArchivo;
                $adj->filename = $nombreLimpio;
                $adj->save();
            }
        }
    }
}

