<?php

namespace FacturaScripts\Plugins\Partes\Lib;

use FacturaScripts\Dinamic\Model\AttachedFile;
use FacturaScripts\Core\Base\DataBase\DataBaseWhere;
use FacturaScripts\Core\Tools;

/**
 * Simple helper to manage file uploads for Partes plugin.
 * It generates unique filenames and avoids saving duplicate
 * records for the same entity.
 */
class UploadManager
{
    /**
     * Handles the upload of multiple files.
     *
     * @param array|null $files The $_FILES entry or null if no files were uploaded.
     * @param string $column    Name of the column in attached_files table.
     * @param int $id           Identifier of the related record.
     */
    public static function upload(?array $files, string $column, int $id): int
    {
        if (empty($files) || !is_array($files['tmp_name'])) {
            return 0;
        }

        $folder = FS_FOLDER . '/MyFiles/';
        $saved = 0;

        foreach ($files['tmp_name'] as $index => $tmpName) {
            if (!is_uploaded_file($tmpName)) {
                self::debugLog('tmp-no-upload', [
                    'tmp' => $tmpName,
                    'column' => $column,
                    'id' => $id,
                    'name' => $files['name'][$index] ?? null,
                    'error' => $files['error'][$index] ?? null,
                    'size' => $files['size'][$index] ?? null,
                ]);
                continue;
            }

            $original = $files['name'][$index];
            $clean = preg_replace('/[^a-zA-Z0-9]+/', '_', $original);
            $clean = trim($clean, '_');

            // Skip if a file with the same name already exists for this record.
            $existing = new AttachedFile();
            $where = [
                new DataBaseWhere($column, $id),
                new DataBaseWhere('filename', $clean),
            ];
            if ($existing->all($where)) {
                self::debugLog('skip-duplicate', [
                    'column' => $column,
                    'id' => $id,
                    'original' => $original,
                    'clean' => $clean,
                ]);
                continue;
            }

            $extension = pathinfo($original, PATHINFO_EXTENSION);
            $tempFile = uniqid('partes_', true) . ($extension ? '.' . $extension : '');
            if (!move_uploaded_file($tmpName, $folder . $tempFile)) {
                self::debugLog('move-failed', [
                    'tmp' => $tmpName,
                    'target' => $folder . $tempFile,
                    'column' => $column,
                    'id' => $id,
                    'original' => $original,
                    'clean' => $clean,
                ]);
                continue;
            }

            $attached = new AttachedFile();
            $attached->$column = $id;
            $attached->filename = $clean;
            $attached->path = $tempFile;
            $attached->save();
            $saved++;

            self::debugLog('move-success', [
                'idfile' => $attached->idfile ?? null,
                'column' => $column,
                'entity_id' => $id,
                'original' => $original,
                'stored' => $tempFile,
            ]);
        }

        return $saved;
    }

    private static function debugLog(string $event, array $context = []): void
    {
        try {
            $payload = json_encode($context, JSON_UNESCAPED_UNICODE | JSON_PARTIAL_OUTPUT_ON_ERROR);
        } catch (\Throwable $exception) {
            $payload = 'json-error: ' . $exception->getMessage();
        }

        Tools::log('partes')->warning('[UploadDebug] ' . $event . ' ' . $payload);
    }
}
