<?php
namespace FacturaScripts\Plugins\NextCloud\Service;

/**
 * Simple helper to interact with a Nextcloud server using WebDAV.
 */
class NextcloudService
{
    /** @var string */
    private $baseUrl;
    /** @var string */
    private $user;
    /** @var string */
    private $token;

    public function __construct(string $baseUrl, string $user, string $token)
    {
        $this->baseUrl = rtrim($baseUrl, '/');
        $this->user = $user;
        $this->token = $token;
    }

    private function buildUrl(string $path): string
    {
        $path = ltrim($path, '/');
        $encodedPath = implode('/', array_map('rawurlencode', explode('/', $path)));
        return $this->baseUrl . '/' . $encodedPath;
    }

    /**
     * Performs a request to the WebDAV API.
     */
    private function request(string $method, string $path, array $headers = [], $body = null): ?string
    {
        $url = $this->buildUrl($path);
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_USERPWD, $this->user . ':' . $this->token);
        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
        if (null !== $body) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        curl_close($ch);
        if (false === $response || $httpCode >= 400) {
            return null;
        }

        return $response;
    }

    /**
     * Moves or renames a remote file/folder.
     * Returns true on success (201/204), false otherwise.
     */
    public function moveFile(string $sourcePath, string $destinationPath, bool $overwrite = false, ?int &$httpCode = null): bool
    {
        $sourceUrl = $this->buildUrl($sourcePath);
        $destUrl = $this->buildUrl($destinationPath);

        $headers = [
            'Destination: ' . $destUrl,
            'Overwrite: ' . ($overwrite ? 'T' : 'F'),
        ];

        $ch = curl_init($sourceUrl);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'MOVE');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_USERPWD, $this->user . ':' . $this->token);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $httpCode = (int)$code;
        return in_array($httpCode, [201, 204], true);
    }

    /**
     * Returns the list of files and folders inside the given path.
     *
     * @return array<array{name:string,isDir:bool,size:int}>
     */
    public function listFiles(string $path = '/'): array
    {
        $xml = $this->request('PROPFIND', $path, ['Depth: 1']);
        if (empty($xml)) {
            return [];
        }

        $doc = @simplexml_load_string($xml);
        if (false === $doc) {
            return [];
        }

        $doc->registerXPathNamespace('d', 'DAV:');
        $doc->registerXPathNamespace('oc', 'http://owncloud.org/ns');
        $doc->registerXPathNamespace('nc', 'http://nextcloud.org/ns');

        $items = [];
        $responses = $doc->xpath('//d:response');
        if (is_array($responses)) {
            array_shift($responses);
        } else {
            $responses = [];
        }
        foreach ($responses as $node) {
            $href = (string)($node->xpath('d:href')[0] ?? '');
            $name = urldecode(basename(rtrim($href, '/')));
            if ($name === '') {
                continue;
            }

            $isDir = !empty($node->xpath('d:propstat/d:prop/d:resourcetype/d:collection'));
            $size = (int)($node->xpath('d:propstat/d:prop/d:getcontentlength')[0] ?? 0);
            if ($isDir) {
                $sizeArray = $node->xpath('d:propstat/d:prop/oc:size')
                    ?: $node->xpath('d:propstat/d:prop/nc:size')
                    ?: $node->xpath('d:propstat/d:prop/d:quota-used-bytes');
                $size = (int)($sizeArray[0] ?? 0);
            }

            $modified = (string)($node->xpath('d:propstat/d:prop/d:getlastmodified')[0] ?? '');
            $mtime = $modified ? strtotime($modified) : 0;

            $items[] = [
                'name'   => $name,
                'isDir'  => $isDir,
                'size'   => $size,
                'mtime'  => $mtime,
            ];
        }

        usort($items, function ($a, $b) {
            if ($a['isDir'] === $b['isDir']) {
                return strnatcasecmp($a['name'], $b['name']);
            }
            return $a['isDir'] ? -1 : 1;
        });

        return $items;
    }

    /**
     * Returns the raw contents of the given file path.
     */
    public function getFile(string $path): ?string
    {
        $content = $this->request('GET', $path);
        if (null !== $content) {
            return $content;
        }

        $localPath = FS_FOLDER . '/' . ltrim($path, '/');
        if (file_exists($localPath)) {
            $localContent = file_get_contents($localPath);
            if (false !== $localContent) {
                return $localContent;
            }
        }

        return null;
    }

    /**
     * Uploads or replaces a file on the Nextcloud server.
     */
    public function putFile(string $path, string $contents): bool
    {
        return null !== $this->request('PUT', $path, [], $contents);
    }

    /**
     * Creates a directory on the Nextcloud server.
     */
    public function makeDir(string $path): bool
    {
        $path = rtrim($path, '/') . '/';
        return null !== $this->request('MKCOL', $path);
	}

    /**
     * Deletes a file on the Nextcloud server.
     */
    public function deleteFile(string $path): bool
    {
        return null !== $this->request('DELETE', $path);
    }
}
