<?php

namespace FacturaScripts\Plugins\NextCloud\Helper;

use FacturaScripts\Core\Base\DataBase\DataBaseWhere;
use FacturaScripts\Core\Tools;
use FacturaScripts\Dinamic\Model\AttachedFileRelation;
use FacturaScripts\Plugins\NextCloud\Service\NextcloudConnectionFactory;

class NextcloudEntityFolderMoveHelper
{
    public static function normalizeFolderValue(string $folder): string
    {
        $clean = trim($folder);
        if ($clean === '') {
            return '';
        }

        $clean = '/' . ltrim($clean, '/');
        $clean = preg_replace('#/+#', '/', $clean ?? '/');
        return rtrim($clean, '/') ?: '/';
    }

    public static function normalizePath(string $path): string
    {
        $clean = trim($path);
        if ($clean === '') {
            return '';
        }

        $clean = '/' . ltrim($clean, '/');
        $clean = preg_replace('#/+#', '/', $clean ?? '/');
        return $clean;
    }

    public static function moveAttachedFiles(string $modelName, $code, string $targetFolder): void
    {
        $targetFolder = self::normalizeFolderValue($targetFolder);
        if ($targetFolder === '') {
            return;
        }

        $where = [new DataBaseWhere('model', $modelName)];
        $where[] = is_numeric($code) ?
            new DataBaseWhere('modelid|modelcode', $code) :
            new DataBaseWhere('modelcode', $code);

        $relationModel = new AttachedFileRelation();
        foreach ($relationModel->all($where, [], 0, 0) as $relation) {
            $file = $relation->getFile();
            if (empty($file->idfile) || empty($file->nextcloud_path)) {
                continue;
            }

            $owner = trim((string)($file->nextcloud_owner ?? ''));
            if ($owner === '') {
                Tools::log()->warning('nextcloud-missing-owner', ['%idfile%' => $file->idfile]);
                continue;
            }

            $service = NextcloudConnectionFactory::buildForNick($owner);
            if (null === $service) {
                Tools::log()->warning('nextcloud-missing-credentials', ['%nick%' => $owner]);
                continue;
            }

            $currentPath = self::normalizePath((string)$file->nextcloud_path);
            $currentName = basename(rtrim($currentPath, '/'));
            if ($currentPath === '' || $currentName === '') {
                continue;
            }

            $destinationPath = NextcloudAttachmentHelper::buildRemotePath($targetFolder, $currentName, (int)$file->idfile);
            if ($destinationPath === $currentPath) {
                continue;
            }

            NextcloudAttachmentHelper::ensureRemotePath($service, dirname($destinationPath));

            $httpCode = null;
            if ($service->moveFile($currentPath, $destinationPath, false, $httpCode)) {
                $file->nextcloud_path = $destinationPath;
                $file->save();
                continue;
            }

            // If destination exists, retry with auto-rename.
            if ((int)$httpCode === 412) {
                $renamed = self::moveWithAutoRename($service, $file, $currentPath, $targetFolder, $currentName);
                if ($renamed) {
                    continue;
                }
            }

            Tools::log()->warning('nextcloud-move-failed', [
                '%idfile%' => $file->idfile,
                '%from%'   => $currentPath,
                '%to%'     => $destinationPath,
                '%code%'   => (int)$httpCode,
            ]);
        }
    }

    private static function moveWithAutoRename($service, $file, string $currentPath, string $targetFolder, string $originalName): bool
    {
        $info = pathinfo($originalName);
        $base = $info['filename'] ?? $originalName;
        $ext = isset($info['extension']) && $info['extension'] !== '' ? '.' . $info['extension'] : '';

        for ($i = 1; $i <= 50; $i++) {
            $candidate = $base . ' (' . $i . ')' . $ext;
            $candidatePath = NextcloudAttachmentHelper::buildRemotePath($targetFolder, $candidate, (int)$file->idfile);
            NextcloudAttachmentHelper::ensureRemotePath($service, dirname($candidatePath));

            $httpCode = null;
            if ($service->moveFile($currentPath, $candidatePath, false, $httpCode)) {
                $file->nextcloud_path = $candidatePath;
                $file->filename = $candidate;
                $file->save();
                return true;
            }

            if ((int)$httpCode !== 412) {
                // Not a name collision, no point retrying.
                return false;
            }
        }

        return false;
    }
}

