<?php

namespace FacturaScripts\Plugins\NextCloud\Helper;

use FacturaScripts\Core\Base\DataBase\DataBaseWhere;
use FacturaScripts\Core\Tools;
use FacturaScripts\Dinamic\Model\AttachedFile;
use FacturaScripts\Dinamic\Model\AttachedFileRelation;
use FacturaScripts\Dinamic\Model\User;
use FacturaScripts\Plugins\NextCloud\Service\NextcloudConnectionFactory;

class NextcloudImportHelper
{
    /**
     * Import files from the root folder assigned to the model and link them as attachments.
     *
     * @param array $selectedNames Optional list of remote file names to import.
     * @return array{imported:int,skipped:int,errors:int,folder:string,status: ?int}
     */
    public static function importForModel(string $modelName, $code, User $user, array $selectedNames = []): array
    {
        $summary = [
            'imported' => 0,
            'skipped' => 0,
            'errors' => 0,
            'folder' => '',
            'status' => null,
        ];
        if (empty($code)) {
            Tools::log()->warning('record-not-found');
            return $summary;
        }

        $service = NextcloudConnectionFactory::buildForUser($user);
        if (null === $service) {
            Tools::log()->warning('nextcloud-missing-credentials');
            return $summary;
        }

        $folder = self::loadExplicitFolder($modelName, $code);
        if ($folder === '') {
            Tools::log()->warning('Nextcloud: asigna una carpeta al registro antes de importar.');
            $summary['folder'] = '';
            return $summary;
        }

        $summary['folder'] = $folder;
        $httpCode = null;
        $items = $service->listFilesWithStatus($folder, $httpCode);
        if (empty($httpCode) || $httpCode >= 400) {
            Tools::log()->warning(sprintf('Nextcloud: no se pudo listar %s (HTTP %d).', $folder, $httpCode));
            $summary['status'] = $httpCode;
            return $summary;
        }

        $summary['status'] = $httpCode;
        $existing = self::existingNextcloudPaths($modelName, $code);
        $selected = self::sanitizeSelectedNames($selectedNames);
        foreach ($items as $item) {
            $remoteName = (string)($item['name'] ?? '');
            if (!empty($selected) && !isset($selected[$remoteName])) {
                continue;
            }

            if (!empty($item['isDir'])) {
                $summary['skipped']++;
                continue;
            }

            $remotePath = self::joinPath($folder, $remoteName);
            if ($remoteName === '' || isset($existing[$remotePath])) {
                $summary['skipped']++;
                continue;
            }

            $content = $service->getFile($remotePath);
            if (null === $content) {
                $summary['errors']++;
                continue;
            }

            $localName = self::normalizeFileName($remoteName);
            if ($localName === '') {
                $summary['errors']++;
                continue;
            }

            $relativePath = self::storeTempFile($localName, $content);
            if ($relativePath === '') {
                $summary['errors']++;
                continue;
            }

            $file = new AttachedFile();
            $file->path = $relativePath;
            if (false === $file->save()) {
                $summary['errors']++;
                continue;
            }

            $file->nextcloud_path = $remotePath;
            $file->nextcloud_owner = $user->nick;
            if (false === $file->save()) {
                $summary['errors']++;
                continue;
            }

            $relation = new AttachedFileRelation();
            $relation->idfile = $file->idfile;
            $relation->model = $modelName;
            $relation->modelcode = (string)$code;
            $relation->modelid = (int)$code;
            $relation->nick = $user->nick;
            if (false === $relation->save()) {
                $summary['errors']++;
                continue;
            }

            $existing[$remotePath] = true;
            $summary['imported']++;
        }

        return $summary;
    }

    /**
     * List files available for import in the assigned root folder.
     *
     * @return array{folder:string,status:?int,items:array<int,array{name:string,path:string,size:int,mtime:int,exists:bool}>,error:string}
     */
    public static function previewForModel(string $modelName, $code, User $user): array
    {
        $preview = [
            'folder' => '',
            'status' => null,
            'items' => [],
            'error' => '',
        ];
        if (empty($code)) {
            $preview['error'] = 'record-not-found';
            return $preview;
        }

        $service = NextcloudConnectionFactory::buildForUser($user);
        if (null === $service) {
            $preview['error'] = 'nextcloud-missing-credentials';
            return $preview;
        }

        $folder = self::loadExplicitFolder($modelName, $code);
        if ($folder === '') {
            $preview['error'] = 'nextcloud-missing-folder';
            return $preview;
        }

        $preview['folder'] = $folder;
        $httpCode = null;
        $items = $service->listFilesWithStatus($folder, $httpCode);
        if (empty($httpCode) || $httpCode >= 400) {
            $preview['status'] = $httpCode;
            $preview['error'] = 'nextcloud-list-failed';
            return $preview;
        }

        $preview['status'] = $httpCode;
        $existing = self::existingNextcloudPaths($modelName, $code);
        foreach ($items as $item) {
            if (!empty($item['isDir'])) {
                continue;
            }

            $remoteName = (string)($item['name'] ?? '');
            if ($remoteName === '') {
                continue;
            }

            $remotePath = self::joinPath($folder, $remoteName);
            $preview['items'][] = [
                'name' => $remoteName,
                'path' => $remotePath,
                'size' => (int)($item['size'] ?? 0),
                'mtime' => (int)($item['mtime'] ?? 0),
                'exists' => isset($existing[$remotePath]),
            ];
        }

        return $preview;
    }

    private static function loadExplicitFolder(string $modelName, $code): string
    {
        $className = '\\FacturaScripts\\Dinamic\\Model\\' . $modelName;
        if (false === class_exists($className)) {
            return '';
        }

        $model = new $className();
        if (false === $model->loadFromCode($code)) {
            return '';
        }

        $folder = trim((string)($model->nextcloud_folder ?? ''));
        if ($folder === '') {
            return '';
        }

        return '/' . ltrim($folder, '/');
    }

    /**
     * @return array<string,bool>
     */
    private static function existingNextcloudPaths(string $modelName, $code): array
    {
        $where = [new DataBaseWhere('model', $modelName)];
        $where[] = is_numeric($code) ?
            new DataBaseWhere('modelid|modelcode', $code) :
            new DataBaseWhere('modelcode', $code);

        $paths = [];
        $relationModel = new AttachedFileRelation();
        foreach ($relationModel->all($where, [], 0, 0) as $relation) {
            $file = $relation->getFile();
            if (!empty($file->nextcloud_path)) {
                $paths[(string)$file->nextcloud_path] = true;
            }
        }

        return $paths;
    }

    /**
     * @return array<string,bool>
     */
    private static function sanitizeSelectedNames(array $names): array
    {
        $selected = [];
        foreach ($names as $name) {
            $name = trim((string)$name);
            if ($name === '' || strpos($name, '/') !== false || strpos($name, '\\') !== false) {
                continue;
            }
            $selected[$name] = true;
        }

        return $selected;
    }

    private static function joinPath(string $folder, string $name): string
    {
        $folder = rtrim($folder, '/');
        $name = ltrim($name, '/');
        return ($folder === '' ? '' : $folder) . '/' . $name;
    }

    private static function normalizeFileName(string $name): string
    {
        $clean = basename(str_replace('\\', '/', $name));
        $clean = Tools::noHtml($clean);
        $clean = trim((string)$clean);
        if ($clean === '') {
            return '';
        }

        $clean = strtolower($clean);
        if (strlen($clean) <= AttachedFile::MAX_FILENAME_LEN) {
            return $clean;
        }

        $parts = explode('.', $clean);
        $extension = count($parts) > 1 ? '.' . array_pop($parts) : '';
        $max = AttachedFile::MAX_FILENAME_LEN - strlen($extension);
        if ($max <= 0) {
            return substr($clean, 0, AttachedFile::MAX_FILENAME_LEN);
        }

        return substr(implode('.', $parts), 0, $max) . $extension;
    }

    private static function storeTempFile(string $name, string $contents): string
    {
        $folder = Tools::folder('MyFiles');
        if (false === Tools::folderCheckOrCreate($folder)) {
            return '';
        }

        $path = $folder . DIRECTORY_SEPARATOR . $name;
        if (file_exists($path)) {
            $path = self::uniquePath($folder, $name);
        }

        if (false === file_put_contents($path, $contents)) {
            return '';
        }

        return basename($path);
    }

    private static function uniquePath(string $folder, string $name): string
    {
        $info = pathinfo($name);
        $base = $info['filename'] ?? $name;
        $ext = isset($info['extension']) && $info['extension'] !== '' ? '.' . $info['extension'] : '';

        for ($i = 1; $i <= 50; $i++) {
            $candidate = $base . ' (' . $i . ')' . $ext;
            $candidatePath = $folder . DIRECTORY_SEPARATOR . $candidate;
            if (!file_exists($candidatePath)) {
                return $candidatePath;
            }
        }

        return $folder . DIRECTORY_SEPARATOR . $name;
    }
}
