<?php

namespace FacturaScripts\Plugins\NextCloud\Extension\Controller;

use Closure;
use FacturaScripts\Core\Base\DataBase\DataBaseWhere;
use FacturaScripts\Core\Tools;
use FacturaScripts\Dinamic\Lib\AssetManager;
use FacturaScripts\Dinamic\Model\AttachedFile;
use FacturaScripts\Core\Model\AttachedFileRelation;
use FacturaScripts\Plugins\NextCloud\Helper\NextcloudFolderHelper;
use FacturaScripts\Plugins\NextCloud\Helper\NextcloudImportHelper;

class EditProyecto
{
    public function createViews(): Closure
    {
        return function (): void {
            if (!isset($this->views['docfiles'])) {
                $this->addHtmlView('docfiles', 'Tab/DocFiles', 'AttachedFileRelation', 'files', 'fas fa-paperclip');
            }
        };
    }

    public function execPreviousAction(): Closure
    {
        return function ($action): bool {
            $activeTab = $this->request->request->get('activetab', $this->request->query->get('activetab', ''));
            $isDocfilesTab = ($activeTab === 'docfiles');

            $validateToken = function (): bool {
                $token = $this->request->request->get('multireqtoken', '');
                if (empty($token) || false === $this->multiRequestProtection->validate($token)) {
                    Tools::log()->warning('invalid-request');
                    return false;
                }

                if ($this->multiRequestProtection->tokenExist($token)) {
                    Tools::log()->warning('duplicated-request');
                    return false;
                }

                return true;
            };

            switch ($action) {
                case 'add-file':
                    if (!$isDocfilesTab) {
                        return true;
                    }
                    if (false === $this->permissions->allowUpdate) {
                        Tools::log()->warning('not-allowed-modify');
                        return true;
                    }
                    if (false === $validateToken()) {
                        return true;
                    }

                    $uploadFiles = $this->request->files->get('new-files', []);
                    foreach ($uploadFiles as $uploadFile) {
                        if ($uploadFile->move(FS_FOLDER . DIRECTORY_SEPARATOR . 'MyFiles', $uploadFile->getClientOriginalName())) {
                            $newFile = new AttachedFile();
                            $newFile->path = $uploadFile->getClientOriginalName();
                            if (false === $newFile->save()) {
                                Tools::log()->error('fail');
                                return true;
                            }

                            $fileRelation = new AttachedFileRelation();
                            $fileRelation->idfile = $newFile->idfile;
                            $fileRelation->model = $this->getModelClassName();
                            $fileRelation->modelcode = $this->request->query->get('code');
                            $fileRelation->modelid = (int)$fileRelation->modelcode;
                            $fileRelation->nick = $this->user->nick;
                            $fileRelation->observations = $this->request->request->get('observations');
                            $this->pipeFalse('addFileAction', $fileRelation, $this->request);

                            if (false === $fileRelation->save()) {
                                Tools::log()->error('fail-relation');
                                return true;
                            }
                        }
                    }

                    Tools::log()->notice('record-updated-correctly');
                    return true;

                case 'delete-file':
                    if (!$isDocfilesTab) {
                        return true;
                    }
                    if (false === $this->permissions->allowDelete) {
                        Tools::log()->warning('not-allowed-delete');
                        return true;
                    }
                    if (false === $validateToken()) {
                        return true;
                    }

                    $fileRelation = new AttachedFileRelation();
                    $id = $this->request->request->get('id');
                    if (false === $fileRelation->loadFromCode($id)) {
                        Tools::log()->warning('record-not-found');
                        return true;
                    }

                    if ($fileRelation->modelcode != $this->request->query->get('code') ||
                        $fileRelation->model !== $this->getModelClassName()) {
                        Tools::log()->warning('not-allowed-delete');
                        return true;
                    }

                    $file = $fileRelation->getFile();
                    $fileRelation->delete();
                    $file->delete();
                    Tools::log()->notice('record-deleted-correctly');
                    return true;

                case 'edit-file':
                    if (!$isDocfilesTab) {
                        return true;
                    }
                    if (false === $this->permissions->allowUpdate) {
                        Tools::log()->warning('not-allowed-modify');
                        return true;
                    }
                    if (false === $validateToken()) {
                        return true;
                    }

                    $fileRelation = new AttachedFileRelation();
                    $id = $this->request->request->get('id');
                    if (false === $fileRelation->loadFromCode($id)) {
                        Tools::log()->warning('record-not-found');
                        return true;
                    }

                    if ($fileRelation->modelcode != $this->request->query->get('code') ||
                        $fileRelation->model !== $this->getModelClassName()) {
                        Tools::log()->warning('not-allowed-modify');
                        return true;
                    }

                    $fileRelation->observations = $this->request->request->get('observations');
                    $this->pipeFalse('editFileAction', $fileRelation, $this->request);

                    if (false === $fileRelation->save()) {
                        Tools::log()->error('record-save-error');
                        return true;
                    }

                    Tools::log()->notice('record-updated-correctly');
                    return true;

                case 'unlink-file':
                    if (!$isDocfilesTab) {
                        return true;
                    }
                    if (false === $this->permissions->allowUpdate) {
                        Tools::log()->warning('not-allowed-modify');
                        return true;
                    }
                    if (false === $validateToken()) {
                        return true;
                    }

                    $fileRelation = new AttachedFileRelation();
                    $id = $this->request->request->get('id');
                    if ($fileRelation->loadFromCode($id)) {
                        $fileRelation->delete();
                    }

                    Tools::log()->notice('record-updated-correctly');
                    return true;

                case 'nextcloud-import':
                    if (!$isDocfilesTab) {
                        return true;
                    }
                    if (false === $this->permissions->allowUpdate) {
                        Tools::log()->warning('not-allowed-modify');
                        return true;
                    }
                    if (false === $validateToken()) {
                        return true;
                    }

                    $code = $this->request->get('code');
                    $selected = $this->request->request->get('nextcloud_files', []);
                    if (!is_array($selected)) {
                        $selected = [$selected];
                    }
                    $summary = NextcloudImportHelper::importForModel(
                        $this->getModelClassName(),
                        $code,
                        $this->user,
                        $selected
                    );

                    $folderInfo = $summary['folder'] !== '' ? ' (' . $summary['folder'] . ')' : '';
                    Tools::log()->notice(sprintf(
                        'Importación Nextcloud%s: %d añadidos, %d omitidos, %d errores.',
                        $folderInfo,
                        $summary['imported'],
                        $summary['skipped'],
                        $summary['errors']
                    ));

                    return true;

                case 'nextcloud-preview':
                    if (!$isDocfilesTab) {
                        return true;
                    }
                    if (false === $this->permissions->allowUpdate) {
                        Tools::log()->warning('not-allowed-modify');
                        return true;
                    }
                    if (false === $validateToken()) {
                        return true;
                    }

                    $code = $this->request->get('code');
                    $preview = NextcloudImportHelper::previewForModel(
                        $this->getModelClassName(),
                        $code,
                        $this->user
                    );
                    if (isset($this->views['docfiles'])) {
                        $this->views['docfiles']->settings['nextcloudPreview'] = $preview;
                    }
                    return true;
            }

            return true;
        };
    }

    public function loadData(): Closure
    {
        return function ($viewName, $view) {
            if ($viewName === 'docfiles') {
                $model = $this->getModelClassName();
                $modelid = $this->request->get('code');
                $where = [new DataBaseWhere('model', $model)];
                $where[] = is_numeric($modelid) ?
                    new DataBaseWhere('modelid|modelcode', $modelid) :
                    new DataBaseWhere('modelcode', $modelid);
                $view->loadData('', $where, ['creationdate' => 'DESC']);
                return;
            }

            AssetManager::addJs(FS_ROUTE . '/Plugins/NextCloud/Assets/JS/nextcloud-folder-selector.js');

            $column = $view->columnForName('nextcloud-folder');
            if (null === $column || $column->widget->getType() !== 'select') {
                return;
            }

            $folders = NextcloudFolderHelper::listRootFolders($this->user);
            $current = trim((string)($view->model->nextcloud_folder ?? ''));
            if ($current !== '' && !isset($folders[$current])) {
                $folders[$current] = $current;
            }
            $column->widget->setValuesFromArray($folders, false, true);
        };
    }
}
