<?php
namespace FacturaScripts\Plugins\NextCloud\Controller;

use FacturaScripts\Core\Base\Controller;
use FacturaScripts\Core\Tools;
use FacturaScripts\Plugins\NextCloud\Service\NextcloudConnectionFactory;
use FacturaScripts\Plugins\NextCloud\Service\NextcloudService;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller with the logic to explore files stored on a Nextcloud instance.
 */
class Nextcloud extends Controller
{
    public $files = [];
    public $path = '/';
    public $parent = '/';
    public $root = '/';

    public function getPageData(): array
    {
        $data = parent::getPageData();
        $data['title'] = 'nextcloud';
        $data['icon'] = 'fas fa-cloud';
        $data['menu'] = 'Cloud';
        $data['showonmenu'] = false;
        return $data;
    }

    protected function getRootPath($user): string
    {
        return '/';
    }

    public function privateCore(&$response, $user, $permissions)
    {
        parent::privateCore($response, $user, $permissions);

        $service = NextcloudConnectionFactory::buildForUser($user);
        if (null === $service) {
            self::toolBox()->i18nLog()->warning('nextcloud-missing-credentials');
            $this->setTemplate(false);
            return;
        }

        $this->root = $this->getRootPath($user);
        if ($this->root !== '/') {
            $service->makeDir($this->root);
        }
      
        $defaultPath = $this->root;
        $path = $this->request->get('path', $defaultPath);
        if (strpos($path, $defaultPath) !== 0) {
            $path = $defaultPath;
        }
        if ($this->request->get('action') === 'download') {
            $this->sendFile($service, $response, $path);
            return;
        }

        if ($this->request->get('action') === 'upload') {
            $upload = $this->request->files->get('file');
            if ($upload && $upload->isValid()) {
                $dest = rtrim($path, '/') . '/' . $upload->getClientOriginalName();
                $service->putFile($dest, file_get_contents($upload->getRealPath()));
            }
            $this->redirect($this->url() . '?path=' . urlencode($path));
            return;
        }

        $files  = $service->listFiles($path);
        $parent = rtrim(dirname($path), '/');
        if ($parent === '' || strpos($parent, $this->root) !== 0) {
            $parent = $this->root;
        }

        $this->path   = $path;
        $this->parent = $parent;
        $this->files  = $files;
        $this->setTemplate('Nextcloud');
    }

    protected function sendFile(NextcloudService $service, Response &$response, string $path): void
    {
        $content = $service->getFile($path);
        if (null === $content) {
            $this->setTemplate(false);
            return;
        }

        $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        $inlineExt = ['pdf', 'png', 'jpg', 'jpeg', 'gif', 'webp'];
        $mimes = [
            'pdf'  => 'application/pdf',
            'png'  => 'image/png',
            'jpg'  => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'gif'  => 'image/gif',
            'webp' => 'image/webp',
        ];

        $response->headers->set('Content-Type', $mimes[$ext] ?? 'application/octet-stream');
        $disposition = in_array($ext, $inlineExt, true) ? 'inline' : 'attachment';
        $response->headers->set('Content-Disposition', $disposition . '; filename="' . basename($path) . '"');
        $response->setContent($content);
        $this->setTemplate(false);
    }
}
