(function () {
    'use strict';

    /**
     * Creates a minimal modal with a lazy-loaded tree of folders.
     */
    function createModal() {
        let modal = document.getElementById('nc-folder-modal');
        if (modal) {
            return modal;
        }

        modal = document.createElement('div');
        modal.id = 'nc-folder-modal';
        modal.className = 'nc-folder-modal';
        modal.innerHTML = '' +
            '<div class="nc-folder-backdrop" data-dismiss="nc-folder-modal"></div>' +
            '<div class="nc-folder-dialog">' +
            '  <div class="nc-folder-header">' +
            '    <strong>Selecciona carpeta</strong>' +
            '    <button type="button" class="close" data-dismiss="nc-folder-modal" aria-label="Close">&times;</button>' +
            '  </div>' +
            '  <div class="nc-folder-body">' +
            '    <div class="nc-folder-status text-muted small mb-2"></div>' +
            '    <div class="nc-folder-tree"></div>' +
            '  </div>' +
            '  <div class="nc-folder-footer text-right">' +
            '    <button type="button" class="btn btn-sm btn-secondary mr-2" data-dismiss="nc-folder-modal">Cancelar</button>' +
            '    <button type="button" class="btn btn-sm btn-primary nc-folder-accept">Usar esta carpeta</button>' +
            '  </div>' +
            '</div>';

        const style = document.createElement('style');
        style.textContent = '' +
            '.nc-folder-modal{position:fixed;top:0;left:0;right:0;bottom:0;display:none;z-index:1050;}' +
            '.nc-folder-backdrop{position:absolute;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,.4);}' +
            '.nc-folder-dialog{position:relative;background:#fff;margin:5% auto;padding:0;max-width:680px;border-radius:4px;box-shadow:0 0 20px rgba(0,0,0,.2);}' +
            '.nc-folder-header{padding:10px 12px;border-bottom:1px solid #e9ecef;display:flex;justify-content:space-between;align-items:center;}' +
            '.nc-folder-body{padding:12px;max-height:420px;overflow:auto;}' +
            '.nc-folder-footer{padding:10px 12px;border-top:1px solid #e9ecef;}' +
            '.nc-folder-tree ul{list-style:none;padding-left:16px;}' +
            '.nc-folder-tree li{margin:2px 0;cursor:pointer;}' +
            '.nc-folder-tree .nc-folder-row{display:flex;align-items:center;}' +
            '.nc-folder-tree .toggle{width:18px;text-align:center;color:#555;cursor:pointer;}' +
            '.nc-folder-tree .name{flex:1;padding:2px 6px;}' +
            '.nc-folder-tree .selected .name{background:#e9f5ff;border-radius:3px;}' +
            '.nc-folder-tree .empty{color:#888;font-style:italic;padding-left:4px;}' +
            '.nc-folder-tree .loading{color:#888;font-style:italic;padding-left:4px;}' +
            '.nc-folder-tree .toggle.disabled{color:#ccc;cursor:default;}' +
            '.nc-folder-dialog .close{border:none;background:transparent;font-size:20px;line-height:1;}' +
            '@media (max-width: 720px){.nc-folder-dialog{margin:10% 12px;}}';
        document.head.appendChild(style);

        document.body.appendChild(modal);

        modal.addEventListener('click', function (event) {
            if (event.target.dataset.dismiss === 'nc-folder-modal' || event.target.classList.contains('nc-folder-backdrop')) {
                hideModal(modal);
            }
        });

        return modal;
    }

    function showModal(modal) {
        modal.style.display = 'block';
    }

    function hideModal(modal) {
        modal.style.display = 'none';
    }

    /**
     * Initializes the tree selector attached to a select[name=nextcloud_folder].
     */
    function initTree(select) {
        if (!select || select.dataset.nextcloudTree === '1') {
            return;
        }
        select.dataset.nextcloudTree = '1';

        const originalValue = (select.value || '').trim() || '/';
        const hidden = document.createElement('input');
        hidden.type = 'hidden';
        hidden.name = select.name;
        hidden.value = originalValue;

        const button = document.createElement('button');
        button.type = 'button';
        button.className = 'btn btn-sm btn-outline-primary mr-2';
        button.textContent = 'Elegir carpeta';

        const current = document.createElement('div');
        current.className = 'text-muted small mt-1';
        current.textContent = 'Seleccionada: ' + hidden.value;

        const container = document.createElement('div');
        container.className = 'nc-folder-picker';
        container.appendChild(button);
        container.appendChild(hidden);
        container.appendChild(current);

        // Sustituimos el select original por el contenedor
        const parent = select.parentNode;
        select.remove();
        if (parent) {
            parent.appendChild(container);
        }

        const modal = createModal();
        const treeContainer = modal.querySelector('.nc-folder-tree');
        const status = modal.querySelector('.nc-folder-status');
        const acceptBtn = modal.querySelector('.nc-folder-accept');
        let selectedPath = hidden.value;

        function setStatus(message, isError) {
            status.textContent = message;
            status.classList.toggle('text-danger', !!isError);
        }

        function renderFolder(parentUl, path, autoSelect) {
            setStatus('Cargando ' + path + ' ...');
            const url = 'NextcloudFolders?path=' + encodeURIComponent(path);
            fetch(url, {credentials: 'same-origin'})
                .then(function (res) {
                    if (!res.ok) {
                        throw new Error('status ' + res.status);
                    }
                    return res.json();
                })
                .then(function (data) {
                    parentUl.innerHTML = '';
                    const folders = data.folders || [];
                    if (folders.length === 0) {
                        const empty = document.createElement('li');
                        empty.className = 'empty';
                        empty.textContent = 'Sin subcarpetas';
                        parentUl.appendChild(empty);
                        return;
                    }

                    folders.forEach(function (folder) {
                        const li = document.createElement('li');
                        li.dataset.path = folder.path;

                        const row = document.createElement('div');
                        row.className = 'nc-folder-row';

                        const toggle = document.createElement('span');
                        toggle.className = 'toggle';
                        toggle.textContent = '▸';

                        const name = document.createElement('span');
                        name.className = 'name';
                        name.textContent = folder.name;

                        row.appendChild(toggle);
                        row.appendChild(name);
                        li.appendChild(row);

                        const children = document.createElement('ul');
                        children.className = 'pl-2';
                        children.style.display = 'none';
                        li.appendChild(children);

                        function selectNode() {
                            selectedPath = folder.path;
                            treeContainer.querySelectorAll('.selected').forEach(function (el) {
                                el.classList.remove('selected');
                            });
                            li.classList.add('selected');
                        }

                        function toggleExpand() {
                            if (children.dataset.loaded === '1') {
                                const visible = children.style.display !== 'none';
                                children.style.display = visible ? 'none' : 'block';
                                toggle.textContent = visible ? '▸' : '▾';
                                return;
                            }

                            children.innerHTML = '<li class="loading">Cargando...</li>';
                            children.style.display = 'block';
                            toggle.textContent = '▾';
                            renderFolder(children, folder.path);
                            children.dataset.loaded = '1';
                        }

                        row.addEventListener('click', function () {
                            selectNode();
                        });

                        toggle.addEventListener('click', function (event) {
                            event.stopPropagation();
                            toggleExpand();
                        });

                        parentUl.appendChild(li);

                        if (autoSelect && selectedPath && selectedPath.indexOf(folder.path) === 0) {
                            selectNode();
                            toggleExpand();
                        }
                    });
                })
                .catch(function () {
                    parentUl.innerHTML = '<li class="empty">No se pudieron cargar las carpetas</li>';
                })
                .finally(function () {
                    setStatus('Carpeta actual: ' + path);
                });
        }

        button.addEventListener('click', function () {
            treeContainer.innerHTML = '<ul></ul>';
            renderFolder(treeContainer.querySelector('ul'), '/', true);
            showModal(modal);
        });

        acceptBtn.addEventListener('click', function () {
            hidden.value = selectedPath || '/';
            current.textContent = 'Seleccionada: ' + hidden.value;
            hideModal(modal);
        });
    }

    document.addEventListener('DOMContentLoaded', function () {
        document.querySelectorAll('select[name="nextcloud_folder"]').forEach(initTree);
    });
})();
